//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.12
import QtQuick.Layouts 1.12

import wesual.Create 1.0
import wesual.Stage  1.0
import wesual.Ui     1.0

UiDialog {
    id : sizeDialog

    property SizeInfo size  : Size.defaultSize()
    property bool acceptableSize : true
    property string infoString

    function transposeSize() {
        const transposed = Size.sizeInfo(size.height, size.width);
        sizeDialog.size = transposed;
    }

    result : size

    onOpened : {
        Creator.context.beginRecordChanges();
    }

    onAccepted : {
        Creator.context.endRecordChanges();
    }

    onRejected : {
        Creator.context.rollbackChanges();
        Creator.context.endRecordChanges();
    }

    contentDelegate : UiTooltipAgent {
        id : dialogContent

        readonly property SizeDatabase db : Creator.context.sizeDatabase

        implicitWidth : 540
        implicitHeight : 280

        Component.onCompleted : sizeView.editSize()

        RowLayout {
            id : content

            anchors.fill : parent
            anchors.margins : 15
            anchors.rightMargin : 4

            spacing : 12

            ColumnLayout {
                id : sizePanel

                readonly property real preferredWidth : 220

                spacing : 4

                Layout.fillWidth : false
                Layout.preferredWidth : preferredWidth
                Layout.alignment : Qt.AlignLeft | Qt.AlignTop

                UiSectionHeader {
                    text : qsTrId("41d01435f6dccfa2")
                }

                Loader {
                    id : sizeView

                    property point editPos

                    function editSize(pos) {
                        editPos = pos || Qt.point(0, 0);
                        sourceComponent = sizeEditorComponent;
                    }

                    sourceComponent : sizeLabelComponent
                    onItemChanged : {
                        if (item.editAt) {
                            item.editAt(editPos);
                        }
                    }

                    Layout.fillWidth : true
                }

                Component {
                    id : sizeLabelComponent

                    RowLayout {
                        Text {
                            font : UiFonts.getFont(UiFonts.Light, 20)
                            text : {
                                let txt = sizeDialog.size.resolution;

                                const rn = sizeDialog.size.resolutionName;
                                if (rn.length > 0) {
                                    txt += " (" + rn + ")"
                                }

                                return txt;
                            }

                            Layout.fillWidth : true

                            MouseArea {
                                anchors.fill : parent
                                cursorShape : Qt.IBeamCursor

                                onClicked : sizeView.editSize(
                                                Qt.point(mouse.x, mouse.y))
                            }
                        }

                        UiToolButton {
                            iconName : "edit"
                            iconSize : UiIcon.Icon_16x16

                            onClicked : sizeView.editSize()
                        }
                    }
                }

                Component {
                    id : sizeEditorComponent

                    SizeEditor {
                        font : UiFonts.getFont(UiFonts.Light, 20)
                        size : sizeDialog.size

                        onEditFinished : {
                            if (size) {
                                sizeDialog.size = size;
                            }
                            sizeView.sourceComponent = sizeLabelComponent;
                        }
                    }
                }

                AspectRatioToggle {
                    id  : toggle
                    size : sizeDialog.size

                    onTransposeOrientation : {
                        dialogContent.forceActiveFocus();
                        sizeDialog.transposeSize();
                    }

                    Layout.bottomMargin : 6
                }

                UiCommandLink {
                    text : qsTrId("fa6a1e8df19ac6e6")

                    onClicked : {
                        dialogContent.forceActiveFocus();
                        sizeList.createCustomSize(sizeDialog.size);
                    }
                }

                Text {
                    id : infoString

                    color : "#333"
                    font : UiFonts.getFont(UiFonts.LightItalic, 14)
                    text : sizeDialog.infoString
                    wrapMode : Text.Wrap

                    Layout.fillWidth : true
                    Layout.topMargin : 6
                }
            }

            UiVerticalSeparator {
                Layout.fillHeight : true
            }

            ColumnLayout {
                spacing : 4

                UiSectionHeader {
                    text : qsTrId("ff001e73c2cf7413")

                    Layout.leftMargin : 4
                }

                SizeList {
                    id : sizeList

                    activeSize : sizeDialog.size
                    sizeDatabase : dialogContent.db

                    onSizePicked : sizeDialog.size = size

                    Layout.fillWidth : true
                    Layout.fillHeight : true
                    Layout.preferredHeight : preferredHeight
                }
            }
        }
    }

    buttons : [
        UiDialogButton {
            role : UiDialogButton.OkRole
            enabled : sizeDialog.acceptableSize
        },
        UiDialogButton {
            role : UiDialogButton.CancelRole
        }
    ]
}
